import { Entity } from '@minecraft/server';

/**
 * The Script API does not have a native way to read keys in .lang files.
 *
 * However, we can use the name of an effect that is translated by the game
 * according to the language to create personalized keys for each language.
 *
 * In this case we are using the `saturation` effect.
 *
 * ---
 *
 * Adding new languages. (Example: Spanish)
 * @example
 * static Saturación = {
 * 	'item.bucketCustomEntity.name': 'Cubo de {0}',
 * };
 */
export class LocalizationParser {
  static OCR_PATTERN = /\{\d+\}/g;

  /** English (United States) */
  static Saturation = {
    'item.bucketCustomEntity.name': 'Bucket of {0}',
  };

  /** Portuguese (Brazil) */
  static Saturação = {
    'item.bucketCustomEntity.name': 'Balde de {0}',
  };

  /**
   * Get the text value from a key.
   * Use {@link LocalizationParser.getParsedTranslatedText} for parse values on text.
   * @param {Entity} emitter The entity to receive the temporary effect.
   * @param {string} key The text translation key.
   * @return {string} Return the translated text.
   */
  static getTranslatedText(emitter, key) {
    emitter.addEffect('saturation', 1, { showParticles: false });
    const effect = emitter.getEffect('saturation').displayName;
    emitter.removeEffect('saturation');

    if (!(effect in LocalizationParser)) return LocalizationParser['Saturation'][key];

    return `§r§f${LocalizationParser[effect][key]}` || undefined;
  }

  /**
   * Get the text value from a key parsing values.
   * @param {Entity} emitter The entity to receive the temporary effect.
   * @param {string} key The text translation key.
   * @return {string} Return the translated text with values. Use {int} to tag values on the string.
   */
  static getParsedTranslatedText(emitter, key, ...values) {
    let message = LocalizationParser.getTranslatedText(emitter, key);
    const matches = message.match(LocalizationParser.OCR_PATTERN);
    const ocrs = matches ? matches.length : 0;

    if (ocrs === 0) return message;

    for (let i = 0; i < ocrs; i++) {
      const ocr = `{${i}}`;
      message = message.replace(ocr, values[i]);
    }

    return message;
  }

  /**
   * Get the text value from a key and remove all the ocurrencies from the string..
   * @param {Entity} emitter The entity to receive the temporary effect.
   * @param {string} key The text translation key.
   * @return {string} Return the translated text with no values.
   */
  static getTranlatedTextAndRemoveOcurrencies(emitter, key) {
    return LocalizationParser.getTranslatedText(emitter, key)
      .replaceAll(LocalizationParser.OCR_PATTERN, '');
  }
}
